namespace java su.artix.accounting.rpc.logic.thriftapi
namespace cpp accounting.logic.thriftapi

// статус карты
enum CardStatus {
    EARN_PAY=0,
    NO_OPERATION=1,
    WITHDRAWAL=2,
    EARN_ONLY=3,
    PAY_ONLY=4
}

// режим работы
enum ServerMode {
    // обычный режим с невозможностью проводить операции с несуществующими номерами карт
    FULL_MODE=0,
    // упрощенный режим, если при проведении операции номер карты не будет найдет, карта будет создана
    LIGHT_MODE=1
}

// тип сервера
enum ServerType {
    // бонусный сервер
    BONUSES=0,
    // сервер сертификатов
    CERTIFICATES=1,
    //комбинированный (сервер сертификатов и бонусный сервер)
    BONUSES_CERTIFICATES=2,
    // сервер купонов
    COUPONS=3
}

// карта
struct Card {
    // номер карты
    1:string cardNumber,
    // статус карты
    2:CardStatus status,
    // номер счета
    3:string accountNumber,
    // дополнительный статус карты
    4:string additionalStatus
}

// возвращаемый результат
struct Result {
    // признак успеха операции
    1:bool success,
    // уникальный идентификатор сессиии
    2:string sessionId,
    // описание результата
    3:string description,
    // баланс
    4:i64 balance,
    // сумма неактивных средств
    5:i64 balanceInactive,
    // номер карты
    6:string cardNumber,
    // номер счета
    7:string accountNumber,
    // статус карты
    8:CardStatus status,
    // дополнительный статус карты
    9:string additionalStatus,
    // информация по группам начислений
    20:list<AssetGroup> assetGroups;
}

// дополнительные данные при создании операции
struct AdditionalAssetInfo {
    // время и дата выполнения операции на кассе
    10: optional i64 dateFromCash,
    // код кассы
    20: optional string cashCode,
    // номер смены
    30: optional string shiftNumber,
    // номер чека
    40: optional string checkNumber,
    // идентификатор родительской операции (если есть). ПАРАМЕТР НЕЛЬЗЯ ПЕРЕДАВАТЬ С КЛИЕНТА, сервер сам его подставляет, если он необходим.
    50: optional string parentSessionId,
    // код магазина
    60: optional string shopCode,
    // источник операции
    70: optional string source,
    // причина операции
    80: optional string reason,
    // pincode для случая когда необходимо проверить валидность карты
    90: optional string pincode;
}

//
struct AssetGroup {
    // идентификатор группы
    10: string groupId,
    // Время начала действия бонусов из этой группы
    20: i64 beginDate,
    // Время окончания действия бонусов из этой группы
    30: i64 endDate,
    // Вес
    40: i32 weight,
    // баланс
    50:i64 balance,
    // сумма неактивных средств
    60:i64 balanceInactive,
    // флаг, явно указывающий, задана ли группа (для C++ имплементации, т.к. с++ в аргументах обязательно должна передавать объект, если он задан)
    70:bool groupIsSet
}

struct Version{
    10: string buildDate,
    20: string version,
    30: string info,
    40: ServerMode serverMode,
    50: ServerType serverType
}
struct NearestFiredBonus {
    // дата окончания бонусов
    1:i64 date,
    // статус карты
    2:i64 count
}

service AccountingLogicService {
    // произвести оплату (оставлено для совместимости со старой системой)
    Result pay(1:string terminalId,2:string sessionId,3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    // отменить оплату (оставлено для совместимости со старой системой)
    Result cancel(1:string terminalId,2:string sessionId,3:i32 status, 4: AdditionalAssetInfo additionalInfo),

    // возврат (оставлено для совместимости со старой системой)
    Result back(1:string terminalId,2:string sessionId,3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    /**
     * получить баланс для карты
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты, по которой нужно получить информацию
     * @param withGroups - возвращать также баланс по группам начисления
     * @return Result - объект, содержащий в себе баланс карты или причину, почему баланс получить не удалось.
     */
    Result balance(1:string terminalId,2:string cardNumber,5:bool withGroups),

    // зачислить  (оставлено для совместимости со старой системой)
    Result add(1:string terminalId,2:string sessionId,3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    // списать  (оставлено для совместимости со старой системой)
    Result sub(1:string terminalId,2:string sessionId,3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    // создать транзакцию по зачислению бонусов
    Result makeEarn(1: string terminalId, 2:string sessionId, 3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    // создать транзакцию по событию возврата товара, за покупку которого начислялись бонусы
    Result makeEarnBack(1: string terminalId, 2:string sessionId, 3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    // создать транзакцию по событию покупки товара за бонусы
    Result makePay(1: string terminalId, 2:string sessionId, 3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    // создать транзакцию по возврату товара, который был куплен за бонусы
    Result makePayBack(1: string terminalId, 2:string sessionId, 3:string cardNumber,4:i64 amount, 5: AdditionalAssetInfo additionalInfo, 6: AssetGroup assetGroup),

    // закрыть транзакцию подтверждением операции
    Result commitTransaction(1:string terminalId, 2:string sessionId, 3:string cardNumber, 10:CardStatus cardStatus, 20:string additionalCardStatus, 30: AdditionalAssetInfo additionalInfo),

    // закрыть транзакцию откатом операции
    Result rollbackTransaction(1:string terminalId,2:string sessionId,3:string cardNumber, 4: AdditionalAssetInfo additionalInfo),

     /**
     * Произвести на сервере операцию, обратную начислению, с привязкой к изначальной sessionId и группе
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты, по которой нужно получить информацию
     * @param sessionId - идентификатор новой операции. Можно не указывать, тогда сервер создаст его на основе UUID
     * @param parentSessionId - идентификатор операции, для группы которой будет создаваться обратная операция. amount обратной операции не должна превышать amount родительской операции
     * @param amount - сумма обратной операции, должна быть не больше суммы родительской операции (если по parentSessionId было начислено 100 бонусов, то списать можно не больше 100 бонусов)
     * @param additionalInfo - дополнительная информация по операции, не обязательна для заполнения.
     * @return Result - результат операции установки баланса.
     */
    Result revertEarnBySessionId(10:string terminalId, 20:string cardNumber, 30:string sessionId, 40:string parentSessionId, 50:i64 amount, 60: AdditionalAssetInfo additionalInfo)

    /**
     * получить информацию о карте с бонусного сервера, если карта не найдена возвратиться объект C ПУСТЫМИ ПОЛЯМИ
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты, по которой нужно получить информацию
     * @return Card - объект карты, если карта не найдена или terminalId не корректен - возвратиться объект с пустыми полями
     */
    Card getCard(10: string terminalId, 20:string cardNumber),


    // проверить существует ли карта. Если существует, в success будет true, остальные поля для данного запроса смысла не несут
    Result cardExists(1:string terminalId,2:Card card),

    /**
     * Установить баланс для карты.
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param sessionId - идентификатор транзацкии
     * @param cardNumber - номер карты, по которой нужно получить информацию
     * @param amount - значение, которое нужно установить, можно передавать отрицательные значения
     * @param ignoreCardStatus - при установке баланса, игнорировать статус карты, то есть оставить возможность начислять и на заблокированную карту
     * @param additionalInfo - дополнительная информация, при установке баланса
     * @param assetGroup - группа, через которую нужно установить баланс
     * @return Result - результат операции установки баланса.
     */
    Result setBalanceForCard(10: string terminalId, 20: string sessionId, 30: string cardNumber, 40:i64 amount, 45: bool ignoreCardStatus, 50: AdditionalAssetInfo additionalInfo, 60: AssetGroup assetGroup),

    /**
     * Получить информацию о версии сервиса
     */
    Version version(),

    /**
     * сменить статус карты
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param card - карта
     * @param status - статус карты
     */
    Result setCardStatus(10:string terminalId, 20:Card card, 30:CardStatus status),

    /**
     * обновить допстатус карты
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param card - карта
     * @param additionalStatus - допстатус карты, для заnullения значения нужно передать пустую строку
     */
    Result setAdditionalCardStatus(10:string terminalId, 20:Card card, 30:string additionalStatus),
    
    /**
     * получить количество и дату сгорания следующих бонусов для карты 
     */
    NearestFiredBonus getNearestFiredBonus(10: string terminalId, 20:string cardNumber)
}

// терминал
struct Terminal {
    // id
    1:string id,
    // признак блокировки
    2:bool blocked
}

struct TransactionStatus {
    // терминал, на котором инициализирована операция
    10:Terminal terminal,
    // статус транзакции
    20:i32 status,
    // дата операции
    30:i64 date,
    //причина, по которой была выполнена данная операция
    40:string reason,
    //источник данных для транзакции
    50:string source
}

// платеж
struct Asset {
    // id
    1:string id,
    // сумма в копейках
    2:i64 amount,
    // статус операции
    3:i32 status,
    // время в милисекундах     // заменить на строку в приемлемом формате
    4:i64 time,
    // терминал
    5:Terminal terminal,
    // статус транзакции
    6:list<TransactionStatus> transactionStatuses,
    // актуальный статус операции
    7: TransactionStatus lastStatus,
    // дополнительные данные
    10: AdditionalAssetInfo additionalInfo,
    // информация о группе проводки
    20: AssetGroup assetGroup
}

// возвращаемый результат сервиса администрации
struct AdminResult {
    1:bool success,
    2:string description
}

// счет
struct Account {
    // номер счета
    1:string  number,
    // список номеров привязанных карт
    2:list<string> cardNumbers
}


// статус терминала
enum TerminalStatus {
    ACTIVE,
    BLOCKED,
    NOT_FOUND
}

// поле, по которому производить сортировку при запросе операций
enum SortAssetsByField {
    // id
    ID,
    // сумма в копейках
    AMOUNT,
    // статус операции
    STATUS,
    // время в милисекундах     // заменить на строку в приемлемом формате
    TIME,
}

// поле, по которому производить сортировку при запросе групп операций
enum SortAssetGroupsByField {
    // groupId
    GROUPID,
    // Время начала действия
    BEGINDATE,
    // Время окончания действия
    ENDDATE,
    // вес
    WEIGHT,
}

//Баланс карты на дату
struct BalanceOnDate {
    //номер счета
    5:string accountNumber
    //номер карты
    10:string cardNumber
    //дата обновления
    20:i64 date
    //баланс
    30:i64 balance
}

//Выполняемый метод
struct InternalMethod {
    //название
    10:string name,
    //описание
    20:string description
}

//ПИН-код для карты
struct Pincode {
    //Номер карты
    10: string number,
    //ПИН-код
    20: string pincode
    // Дата обновления
	30: i64 update_time
}

service AccountingAdminLogicService {

    /**
     * Получить список операций для карты
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты, по которой нужно получить список операций
     * @param onlyForCard - если параметр установлен, будут показаны транзакции только по текущей карте, в общем случае - показываются транзакции по счету
     * @return list<Asset> - список проведенных по карте операций, если операция по к-л причине не удалась - возвратитсся пустой список.
     */
    list<Asset> getAssetList(1:string terminalId,2:string cardNumber, 3:bool onlyForCard),

    /**
     * Получить список операций для карты
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты, по которой нужно получить список операций
     * @param onlyForCard - если параметр установлен, будут показаны транзакции только по текущей карте, в общем случае - показываются транзакции по счету
     * @param fromDate - начало диапазона дат
     * @param toDate - конец диапазона дат
     * @return list<Asset> - список проведенных по карте операций, если операция по к-л причине не удалась - возвратитсся пустой список.
     */
    list<Asset> getAssetListFromDateToDate(1:string terminalId,2:string cardNumber, 3:bool onlyForCard, 4:i64 fromDate, 5:i64 toDate),
    
	list<Asset> getAssetListFromDate(1:string terminalId,2:string cardNumber, 3:bool onlyForCard, 4:i64 fromDate),
	
	list<Asset> getAssetListToDate(1:string terminalId,2:string cardNumber, 3:bool onlyForCard, 4:i64 toDate),
	  
    /**
     * Получить частичный (постраничный) список операций для карты
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты, по которой нужно получить список операций
     * @param pageNumber - номер страницы
     * @param pageSize - количество записей на странице
     * @param sortedByField - по какому полю сортировать записи
     * @param ascending - при установленном ключе - записи будут сортироваться по возрастанию
     * @param onlyForCard - если параметр установлен, будут показаны транзакции только по текущей карте, в общем случае - показываются транзакции по счету
     * @return list<Asset> - список проведенных по карте операций, если операция по к-л причине не удалась - возвратитсся пустой список.
     */
    list<Asset> getPageableAssetList(1:string terminalId,2:string cardNumber,3:i32 pageNumber, 4:i32 pageSize, 5:SortAssetsByField sortedByField, 6:bool ascending,  7:bool onlyForCard),

    /**
    * Создать терминал на БС.
    * @param terminal - объект заводимого терминала
    * @param clusterId - идентификатор кластера, в котором нужно завести терминал, если указан, сервер добавит терминал, если он в кластерном режиме, для создания без указания кластера, в качестве параметра нужно передавать null
    **/
    AdminResult createTerminal(1:Terminal terminal, 2:string clusterId),

    // получить статус терминала
    TerminalStatus getTerminalStatus(1:string terminalId) ,

    // заблокировать терминал
    AdminResult blockTerminal(1:string terminalId),

    // активировать терминал
    AdminResult activateTerminal(1:string terminalId),

    // завести карту
    AdminResult createCard(1:string terminalId,2:Card card),

    /**
     * Получить баланс по набору карт
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumbers - коллекция, состоящая из номеров карт (строки), по которым нужно получить информацию, максимальное количество карт: 100, при большем количестве операция возвратит ошибочный результат
     * @param withGroups - возвращать также баланс по группам начисления
     * @return list<Result> - коллекция типа Result, в которой будет содержаться информация по каждой карте отдельно. Также может содержать только один объект без указания номера карты - значит произошла ошибка
     *                          например, на проверке terminalId
     */
    list<Result> balance(1:string terminalId,2:list<string> cardNumbers,5:bool withGroups),

    // проверить существует ли карта
    AdminResult cardExists(1:string terminalId,2:Card card),

    // сменить статус карты
    AdminResult setCardStatus(1:string terminalId,2:Card card,3:CardStatus status,4:string additionalStatus),

    // получить количество карт на сервере
    i32 getCardCount(1:string terminalId),

    //получить частичный список карт
    list<Card> getCards(1: string terminalId, 2:i32 pageNumber, 3:i32 pageSize,),

    //обновить (или создать) данные об актуальных балансах на картах, при указанных параметрах fromDate и toDate в результат попаду только те счета, которые
    //были изменены (или созданы) в указанный период, если fromDate или toDate равны нулю, считается, что они не указаны
    //НА ТЕКУЩИЙ МОМЕНТ НЕ ВСЕ СГОРАЮЩИЕ БОНУСЫ ТАКЖЕ НЕ БУДУТ ОТРАЖЕНЫ В СПИСКЕ, если будут указаны параметры fromDate или toDate
    AdminResult createActualBonusesTable(10: string terminalId, 20: i64 fromDate, 30: i64 toDate),

    //обновить значения lastKnownBonusBalance и lastKnownBonusBalanceDate в карте сгенерированными функцией createActualBonusesTable() данными
    AdminResult updateActualBonusesOnCard(10: string terminalId),

    //получить количество записей в таблице, сгенерированной функцией createActualBonusesTable(10: string terminalId)
    i64 getActualBonusesTableCount(10: string terminalId),

    //получить порцию записей из таблицы, созданной функцией createActualBonusesTable(10: string terminalId)
    list<BalanceOnDate> getActualBonuses(10: string terminalId, 20:i32 pageNumber, 30:i32 pageSize),

    /**
     * получить информацию о карте с бонусного сервера, если карта не найдена возвратиться объект C ПУСТЫМИ ПОЛЯМИ
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты, по которой нужно получить информацию
     * @return Card - объект карты, если карта не найдена или terminalId не корректен - возвратиться объект с пустыми полями
     */
    Card getCard(10: string terminalId, 20:string cardNumber),

    /*
     * Задать для транзакции индивидуальную причину (3-й параметр - порядковый номер транзакции для данной проводки, отсортированные по дате)
     */
    AdminResult setTransactionReason(10: string terminalId, 20: string sessionId, 30: i32 transactionItem, 40: string reason),

    /**
     * Получить список исполняемых скриптов
     */
    list<InternalMethod> getInternalMethods(10: string terminalId),

    /**
     * Запустить работу скрипта
     */
    AdminResult executeInternalMethod(10: string terminalId, 20: string nameOfMethod),

    //Интерфейс для работы с группами карт

    /**
     * Получить количество записей для групп акций, созданных на бонусном сервере
     * Стоит брать в рассмотрение тот факт, что у групп м.б. быть одинаковый groupId, но различные begin и end, и может получиться,
     * что каждой группе соответствует единственная операция(проводка)
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param beginDateBiggerThan - в выборку попадут только те группы, у которых дата начала больше этого значения, если будет 0 - не учитывать
     * @param endDateLessThan - в выборку попадут только те группы, у которых дата начала больше этого значения, если будет 0 - не учитывать
     * @return i64 - количество объектов, удовлетворяющих условиям
     */
    i64 getAssetGroupsCount(10: string terminalId, 20: i64 beginDateBiggerThan, 30: i64 endDateLessThan),

    /**
     * Получить список групп акций, созданных на бонусном сервере
     * Стоит брать в рассмотрение тот факт, что у групп м.б. быть одинаковый groupId, но различные begin и end, и может получиться,
     * что каждой группе соответствует единственная операция(проводка)
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param beginDateBiggerThan - в выборку попадут только те группы, у которых дата начала больше этого значения, если будет 0 - не учитывать
     * @param endDateLessThan - в выборку попадут только те группы, у которых дата окончания меньше этого значения, если будет 0 - не учитывать
     * @param pageNumber - номер страницы данных, которые нужно отобразить
     * @param pageSize - количество записей в каждой странице
     * @return list - список объектов, удовлетворяющих условиям
     */
    list<AssetGroup> getAssetGroups(10: string terminalId, 20: i64 beginDateBiggerThan, 30: i64 endDateLessThan, 40: i32 pageNumber, 50: i32 pageSize, 60:SortAssetGroupsByField sortedByField, 70:bool ascending),

    /**
     * Получить список групп, в которой присутствует данная карта
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер существующей на БС карты
     * @return list - список объектов, удовлетворяющих условиям
     */
    list<AssetGroup> getAssetGroupForCard(10: string terminalId, 20: string cardNumber),

    /**
     * Изменить параметры группы акций
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param groupId - идентификатор группы акций, у которой нужно поменять параметры
     * @param beginShift - смещение текущего времени начала действия группы акций на заданное количество мс
     * @param endShift - смещение текущего времени окончания действия группы акций на заданное количество мс
     * @param newWeight - задание нового веса для данной группы акций, передача 0 не будет изменять текущее значение
     * @return AdminResult - результат выполнения запроса
     */
    AdminResult changeAssetGroupParams(10: string terminalId, 20: string groupId, 30: i64 beginShift, 40: i64 endShift, 50: i32 newWeight),

    /**
     * Получить информацию о версии сервиса
     */
    Version version(),

    /**
     * Создать счет
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param accountNumber - номер счета
     * @return AdminResult - результат выполнения запроса
     */
    AdminResult createAccount(10:string terminalId, 20:string accountNumber),

    /**
     * Получить информацию о счете
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param accountNumber - номер счета
     * @return Account - информация о счете
     */
    Account getAccount(10:string terminalId, 20:string accountNumber),

    /**
     * Проверить существует ли счет
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param accountNumber - номер счета
     * @return AdminResult - результат выполнения запроса
     */
    AdminResult accountExists(10:string terminalId, 20:string accountNumber),

    /**
     * Привязать карту к счету
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты
     * @param accountNumber - номер счета
     * @param bindForce - Если карта уже привязана к какому-то счету, то перепривязать к новому
     * @return AdminResult - результат выполнения запроса
     */
    AdminResult bindCard(10:string terminalId, 20:string cardNumber, 30:string accountNumber, 40:bool bindForce),

    /**
     * Отвязать карту от счета
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param cardNumber - номер карты
     * @param accountNumber - номер счета
     * @return AdminResult - результат выполнения запроса
     */
    AdminResult unbindCard(10:string terminalId, 20:string cardNumber, 30:string accountNumber)
    
    /**
     * Очистить таблицу с ПИН-кодами
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     */
    AdminResult clearPincodes(10:string terminalId)
    
    /**
     * Удалить ПИН-коды
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param number - список номеров карт, для которых надо удалить ПИН-коды
     */
    AdminResult deletePincodes(10:string terminalId, 20:list<string> number)
    
    /**
     * Добавить ПИН-коды
     * @param terminalId - идентификатор зарегистрированного на БС терминала
     * @param pincodes - список ПИН-кодов, которые надо добавить
     */
    AdminResult addPincodes(10:string terminalId, 20:list<Pincode> pincodes)

}